/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////
// authors     Jesko Nordlohne,  Andy Grabow
// Interpreter fuer ELfe-Roboter
// + Three-Way-Handshake-Kommunikation von AksenBoard nach PC
/////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////

// 2000 mikrosek   => 180 grad

// Implementiertes SAFAA-PROTOKOLL
// PC <==> AKSEN        BIDIRETIONALE KOMMUNIKATION
//    ==>> s			syn Verbindungsaufbau anfordern
//    <<== a			ack Verbindungsaufbau bestaetigen

//    ==>> {1,2,3,}     Kommando-Triplet (Trennzeichen Komma) {Servonr, Winkel, Zeit}
//    <<== a            Empfangenes Triplet bestaetigen (bei ungueltigem Triplet <<== n   und sprung  zu syn

//    ==>> f	 		fin Verbindung terminieren
//    <<== x			ack: Alle Packete empfangen
//    ==>> a			ack erfolgreiche Uebertragung bestaetigen
//    <<== f            board bestaetigt erfolgreiche Programm-Abarbeitung

// ==>> n           n (nack) dient jederzeit als Fehlermeldung an PC

// TODO
// Ausfuehren der interpretierten Befehlsfolge

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "serielle.h"

#define MAXPULS 2100.0
#define MINPULS 500.0
#define MAXZEIT 500	// maximale Dauer eines sleeps nach Absetzen des Servo-Befehls

struct befehl{
	unsigned char servonr;
	unsigned short int winkel;
	unsigned short int zeit;
};


int	arrayindex;

char serielle_getcha(void);
unsigned char abarbeitung(struct befehl *,int);
void putstring(char[]);
void startposition(void);
void servos(unsigned char id, int grad);

xdata struct befehl programm[100];						// Programm aus bis zu 100 Befehlen

void AksenMain(void)
{

	char forbidden_char_received=0,
		zeichen;										// um empfange message zu tokenizen
	unsigned short int servonr=0, winkel=0, zeit=0,		// Servo-Befehle
		zahl=0,											// zur Berechnung der numerischen Werte der Ziffern-Strings
		msg=0;											// Spezifiziert Element der Message

	unsigned char packets=0;							// zaehlt empfangene Pakete (0 bis 255)

	serielle_init(); 									// Verbinung it PC Initialisieren (9600 baud)
	servobank_start();									// Servos initialisieren


	startposition();									// Sartposition einnehmen ...
	putstring("horche com...");							// ... und auf Befehle am COM-Port warten

	while (1){ 											// Endlosschleife zum Lesen von Progs per 3-way-handshake
		forbidden_char_received = 0;					// keine ungueltigen Zeichen in neuem Programm
		packets = 0;									// Anzahl empfangener Packete auf 0 setzen
		msg = 0;										// welcher Teil des Triplets wird gerade empfangen?
		zeichen=serielle_getchar();						// von COM-Port lesen
		if(zeichen=='s'){								// SYN

			serielle_putchar('a');						// VERBINDUNG STEHT, WARTEN AUF TRIPLETS

			putstring("got syn - #?");
			//sleep(1000);


			// Einlesen der Befehlstriplets bis fin (f) empfangen wird
			while( (!forbidden_char_received) && ((zeichen = serielle_getchar()) != 'f')){

				if( (zeichen >= '0') && (zeichen <= '9') )			// Ziffer erkannt
					zahl = 10*zahl + (zeichen - 48);
				else if(zeichen == ',')								// Triplet-Trennzeichen KOMMA
				{
					switch(msg)										// speichere den Wert entsprechend ab
					{
					case 0:
						servonr = zahl;
						break;
					case 1:
						winkel = zahl;
						break;
					case 2:
						zeit = zahl;
						// Hier ist das Befehls-Triplet komplett empfangen
						// Syntaxcheck => ist Individuum defekt?
						if ( (servonr<0) || (servonr>10) || (winkel<0) || (winkel>180) || (zeit<0) || (zeit > MAXZEIT) ){
							forbidden_char_received = 1;
							putstring("defekt-mutation");
							sleep(1000);
						} else{
							programm[packets].servonr = servonr;
							programm[packets].winkel = winkel;
							programm[packets].zeit = zeit;
							serielle_putchar('a');					// Empfang des vollstaendigen Triplets bestaetigen
							packets++;								// Befehl komplett => Packetszhler
						}
						break;
					}

					zahl = 0;										// reset zahl fr Berechnung des naechstes Werts
					msg = (msg + 1)%3;								// Ende dieses Tripletelementes
				}
				else{
					//lcd_cls();
					//lcd_int(zeichen);
					///sleep(5000);
					forbidden_char_received = 1;					// falsches Zeichen empfangen, breche Emfang der Daten ab
				}
			}

			if(zeichen != 'f'){										// Fin (f) wird erwartet - alles andere >> Fehler
				putstring("Progr.fehler");
				serielle_putchar('n');				 				// NACK senden => springt an Anfang der while-Schleife
				sleep(1000);
			}

			if((msg == 0) && (!forbidden_char_received))			// komplette Befehle empfangen?
			{
				serielle_putchar(packets);							// erfolgreichen Paketempfang besttigen
				zeichen=serielle_getchar();							// auf Besttigung warten
				if(zeichen == 'a'){ 								// ACK
					abarbeitung(programm, packets);					// Komplettes Programm (alle Befehle) abarbeiten
				}
				else if(zeichen=='n'){								// empfangene Packete nicht korrekt, Empfang wird abgebrochen
					putstring("ERR 3way handshake");
					sleep(1000);
				}
				else{
					forbidden_char_received = 1;					// Error, bertragung oder Programm fehlerhaft
					putstring("illegal char");
					sleep(1000);
				}
			}
			else {
				putstring("resend command");
				sleep(1000);
			}
		}
		else{
			forbidden_char_received = 1;							// SYN erwartet, breche bertragung ab
			//lcd_cls();
			//lcd_int(zeichen);
			//sleep(5000);
			putstring("syn erwartet");
			serielle_putchar('n');
			sleep(1000);
		}

		// erfolgreiche bertragung, warte auf den nchsten Input
		// zurck zum Anfang
	}
}


unsigned char abarbeitung(struct befehl *programm, int packets){

	int j;
	int k;
	for (k=0; k<3; k++)
		for (j=0; j<packets; j++){

			// Befehl auf LCD ausgeben
			lcd_cls();
			lcd_int(j);
			lcd_puts(" s");
			lcd_int(programm[j].servonr);
			lcd_puts(" w");
			lcd_int(programm[j].winkel);
			lcd_puts(" z");
			lcd_int(programm[j].zeit);

			// und Befehl ausfuehren
			servos(programm[j].servonr, programm[j].winkel);

			// gem Befehl warten
			sleep(programm[j].zeit);
		}

	serielle_putchar('f');							// meldung an pc: programm fertig abgearbeitet
	lcd_puts(":-)");
	return 1;
}


//gibt einen String auf dem Display aus
void putstring(char text[]){
	lcd_cls();
	lcd_puts(text);
}


void servos(unsigned char id, int grad) // erwartet gradzahl (0 bis 180)
{
	// zweites Argument:  Umwandlung gradzahl => pulsweitenmodulation
	if(id < 8) servobank(id, (float)grad/180.0*(MAXPULS-MINPULS)+MINPULS);	// erwartet pwm-wert
	else servo(id-8, (float)grad/180.0*(MAXPULS-MINPULS)+MINPULS);
}



void startposition(void){
	int i;
	for(i=0; i<11; i++){	// alle Servos hinlegen
		servos(i,0);
	}

}
